# Useful functions for action scripts.
#
# $Id$
#
# Copyright 2016-2022, Juniper Networks, Inc
# All Rights Reserved
#

if test -z "$_DEBUG_SH"; then
    if test -z "$_PKG_SUBR_SH$_VDOT_SH"; then
        . ${PKGTOOLS:-/usr/libexec}/debug.sh
    else
        vdot ${PKGTOOLS:-/usr/libexec}/debug.sh
    fi
fi

# be compatible with atexit
Exit() {
    ExitStatus=$1
    exit $1
}

Error() {
    case "$1" in
    [1-9]|[1-9][0-9]) rc=$1; shift;;
    *) rc=1;;
    esac
    echo "ERROR: $@" >&2
    Exit $rc
}

Warning() {
    echo "WARNING: $@" >&2
}

# return the first of $* that exists
# we generally use this to deal with expanding wildcards.
Exists() {
    local _t=-s

    while :
    do
        case "$1" in
        -?) _t=$1; shift;;
        *) break;;
        esac
    done
    for f in $*
    do
        test $_t $f || continue
        echo $f
        return 0
    done
    return 1
}

if ! type readlink > /dev/null 2>&1; then
readlink() {
    'ls' -ld $1 | sed 's,.*-> ,,';
}
fi

read_link() {
    if test -L $1; then
        readlink $1
    else
        echo $1
    fi
}

# override if needed
set_flags() { : ; }

# make/correct a symlink
make_symlink() {
	local src=$1
	local target=$DESTDIR/${2#$DESTDIR/}
	local d=${target%/*}
	local link

	shift 2
	DebugOn make_symlink:$target make_symlink:${target#$DESTDIR}
	if test -d $target -a ! -L $target; then
		# we are making a symlink in a directory!
		# fix target to be what we expect below
		target=$target/${src##*/}
	fi
	if test -d $d; then
		if test -L $target; then
			local PKGMNT=${PKGMNT:-$pkgmnt}
			PKGMNT=${PKGMNT#$DESTDIR}
			link=`read_link $target`
			case "$link" in
			$src)	set_flags -h $target "$@"
				return 0
				;;
			${PKGMNT%-*}-*)
				if test ${PKG_SYMLINK_VERSION:-0} -gt 0; then
					# we are a package that allows
					# for backwards compatability
					# this gets complicated...
					if Exists $link $DESTDIR$link > /dev/null; then
						local link_version=`echo "$link" |
						sed -n "s,.*${PKGMNT%/*}/,,;s,/.*,,;s,.*-\([1-9][0-9]*\)\$,\1,p"`
						test ${link_version:-0} -gt $PKG_SYMLINK_VERSION && return 0
					fi
				fi
				;;
			esac
		fi
		if ! rm -f $target; then
			chflags -h noschg $target
			rm -f $target
		fi
	else
		mkdir -p $d
	fi
	ln -sf ${src#$DESTDIR} $target
	set_flags -h $target "$@"
	DebugOff make_symlink:$target make_symlink:${target#$DESTDIR}
}

# make the symlinks listed on stdin
make_symlinks() {
	DebugOn -o make_symlinks make_symlinks:${PKG_BASENAME} symlinks
	while read src target flags
	do
		case "$src" in
		"#"*) continue;;
		esac
		case "$target" in
		/pkg/*|/scripts/*) continue;;
		*/*) ;;
		*) continue;;
		esac
		make_symlink $src $target $flags
	done
	DebugOff make_symlinks make_symlinks:${PKG_BASENAME} symlinks
}

##
# restore_symlinks
#
# some packages override content of others.
# When they are deactivated (except when being
# replaced by newer version), we want to restore
# symlinks we overrode - provided they still point to us.
#
restore_symlinks() {
    local pkgdir=$1
    local rlinks=$pkgdir/contents/restore.symlinks

    DebugOn restore_symlinks
    if test_verified -s $rlinks; then
	# format of restore.symlinks is
	# basename pattern
	(
	    slinks=/dev/null
	    lbname=/dev/null
	    # ignore symlinks not currently "owned" by this package
	    get_xmltags $pkgdir basename
	    while read bname pattern
	    do
		: bname=$bname pattern=$pattern
		case "$bname" in
		"#"*) continue;;
		$lbname) ;;
		*)
		    # deal with a special case first
		    case "$pattern" in
		    origin-all-contents)
			# all our content originally belonged to $bname
			sed "s,/$basename/,/$bname/," $pkgdir/contents/contents.symlinks
			break	# this should/must be only entry!
			;;
		    esac

		    last=${pkgsmnt:-$PKGSMNT}/$bname.last
		    if test -L $last; then
			iso=`readlink $last`
			slinks=${iso%.*}.symlinks
		    else
			bpd=`egrep "package name=|basename>$bname<" ${pkgsets:-$PKGSETS}/active/packages.xml |
			grep -B1 basename |
			sed -n '/name=/s,.*"\(.*\)".*,\1,p'`
			slinks=${pkgroot:-$PKGROOT}/$bpd/contents/contents.symlinks
		    fi
		    lbname=$bname
		    ;;
		esac
		test -s $slinks || continue
		egrep "$pattern" $slinks
	    done | sort -u |
	    while read src target flags
	    do
		case "$src" in
		"#"*) continue;;
		esac
		case "$target" in
		/pkg/*|/scripts/*) continue;;
		*/*) ;;
		*) continue;;
		esac
		t=$DESTDIR/${target#$DESTDIR}
		test -L $t || continue
		link=`read_link $t`
		case "$link" in
		*/mnt/$basename/*) ;;
		*) continue;;
		esac
		make_symlink $src $target $flags
	    done
	) < $rlinks
    fi
    DebugOff restore_symlinks
}

# remove a symlink if it matches
# this is the opposite of make_symlink
rm_symlink() {
    local src=$1
    local target=$DESTDIR/${2#$DESTDIR}
    local d=${target%/*}
    local link

    shift 2

    DebugOn rm_symlink:$target rm_symlink:${target#$DESTDIR}
    if test -d $d; then
        if test -L $target; then
            link=`read_link $target`
            case "$link" in
            $src) # yes it is ours - remove it
                if ! rm -f $target; then
                    chflags -h noschg $target
                    rm -f $target
                fi
                ;;
            esac
        fi
    fi
    DebugOn rm_symlink:$target rm_symlink:${target#$DESTDIR}
}

# remove the symlinks listed on stdin
rm_symlinks() {
	DebugOn -o rm_symlinks symlinks
	while read src target flags
	do
		case "$src" in
		"#"*) continue;;
		esac
		case "$target" in
		/pkg/*|/scripts/*) continue;;
		*/*) ;;
		*) continue;;
		esac
		rm_symlink $src $target
	done
	DebugOff rm_symlinks symlinks
}

# be sure to unset any variables you expect this to set
get_xmltags() {
    local xmlf=$1; shift
    local e x

    test -s ${xmlf} || xmlf=$DESTDIR$xmlf
    test -d ${xmlf:-/dev/null} && xmlf=$xmlf/package.xml
    if test -s ${xmlf:-/dev/null}; then
        e="$1"; shift
        for x in "$@"
        do
            e="$e|$x"
        done
        # extract the patterns
        # we can handle up to five '-' in tag/toggle name
        # which will be converted to _
        eval `egrep "<($e)>" $xmlf 2> /dev/null |
        sed -e 's,</.*,,;s,<,,;s,/>,=1,;s,>,=,' \
            -e '/^[^=]*-/s,-,_,' \
            -e '/^[^=]*-/s,-,_,' \
            -e '/^[^=]*-/s,-,_,' \
            -e '/^[^=]*-/s,-,_,' \
            -e '/^[^=]*-/s,-,_,'`
    fi
}

clear_features() {
    local fl=`set | egrep '^(pkg_feature|feature)_' | sed 's,=.*,,'`

    test -n "$fl" && unset $fl
}

get_features() {
    # this will set feature_*=1 pkg_feature_*=1
    get_xmltags $1 "feature-.*" "pkg-feature-.*"
}

# get all the feature-* toggles
# once should do
got_features_once=
get_features_once() {
    case ",$got_features_once," in
    *,$1,*) return 0;;
    esac
    got_features_once=$got_features_once,$1
    get_features $1
}

# return true if any of the toggles are present
pkg_has_toggle() {
    local pkgdir=$1; shift
    local t toggles=${1%/}; shift

    for t in "$@"
    do
        toggles="$toggles|${t%/}"
    done

    egrep -q "<($toggles)/>" $pkgdir/package.xml 2> /dev/null
}

# for packages which lack mountable contents
# but have contents.symlinks they are typically
# relative to $pkgdir and we need to add plink
# (/packages/sets/active/$linknname)
# but *only* if the src exists under $pkgdir
pkg_symlinks_filter() {
    local plink pkgdir=$1; shift
    case "$1" in
    $pkgsets/*) plink=$1; shift;;
    *) plink=$pkgsets/active/`_pkg_linkname $pkgdir`;;
    esac

    DebugOn -o pkg_symlinks_filter make_symlinks symlinks
    sed -e "s,%PKGDIR%,$plink," "$@" |
    while read src target
    do
        case "$src" in
        /*) ;;
        *)  # could be relative to $pkgdir ?
            if test -e $pkgdir/$src; then
                src=$plink/$src
            fi
            ;;
        esac
        echo $src $target
    done
    DebugOff pkg_symlinks_filter make_symlinks symlinks
}

# convenience functions for action scripts
is_member() {
    local pkgset=$1
    local pkgdir=$2
    local pkgsets=${pkgsets:-${PKGSETS:-$DESTDIR/packages/sets}}

    # check if we were just given linkname
    test -L $pkgsets/$pkgset/$pkgdir && return 0
    'ls' -l $pkgsets/$pkgset | grep -q ">.*/${pkgdir##*/}"
}

is_active() {
    is_member active $1
}

##
# packages_with_toggle "toggles" package[s].xml ...
#
# Return a list of {link,base}name values
# of packages in packages.xml (or a set of package.xml files)
# that have any of the specified toggles.
# This works because tags are sorted within package.xml
# and entries in packages.xml are sorted in mount order
packages_with_toggle() {
    toggles=$1; shift

    # if we are given individual package.xml files
    # assume they are not in the correct order
    case "$1" in
    *package.xml) # attempt to fix the order
        ordered=`${pkgorder:-pkgorder} "$@" 2> /dev/null`
        test -z "$ordered" || set -- $ordered
        ;;
    esac
    egrep "<(base|link)name>|($toggles)/>" "$@" 2> /dev/null |
    egrep -B1 "($toggles)/>" | sed -n '/name>/s/.*>\([^<]*\)<.*/\1/p'
}

# some devices take a while to show up
waitfor_dev() {
    local _t i dev

    case "$1" in
    -?) _t=$1; shift;;
    *) _t=-c;;
    esac
    for i in 0 0 1 1 1 2 2 3 3 3 
    do 
        for dev in "$@"
        do
            test $_t $dev && return 0
            sleep $i
        done
    done
    return 1
}
